% // ====================================================================
% // This file is part of the Endmember Induction Algorithms Toolbox for MATLAB 
% // Copyright (C) Grupo de Inteligencia Computacional, Universidad del 
% // Pa??s Vasco (UPV/EHU), Spain, released under the terms of the GNU 
% // General Public License.
% //
% // Endmember Induction Algorithms Toolbox is free software: you can redistribute 
% // it and/or modify it under the terms of the GNU General Public License 
% // as published by the Free Software Foundation, either version 3 of the 
% // License, or (at your option) any later version.
% //
% // Endmember Induction Algorithms Toolbox is distributed in the hope that it will
% // be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
% // of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
% // General Public License for more details.
% //
% // You should have received a copy of the GNU General Public License
% // along with Endmember Induction Algorithms Toolbox. 
% // If not, see <http://www.gnu.org/licenses/>.
% // ====================================================================
%% [W,M] = EIA_LAM2(X,Y)
% 
% Manuel Grana <manuel.grana[AT]ehu.es>
% Miguel Angel Veganzones <miguelangel.veganzones[AT]ehu.es>
% Ion Marques <ion.marques[AT]ehu.es>
% Grupo de Inteligencia Computacional (GIC), Universidad del Pais Vasco /
% Euskal Herriko Unibertsitatea (UPV/EHU)
% http://www.ehu.es/computationalintelligence
% 
% Copyright (2011) Grupo de Inteligencia Computacional @ Universidad del Pais Vasco, Spain.
%
% The Lattice Associative Memories is a kind of associative memory working
% over lattice operations. If X=Y then W and M are Lattice AutoAssociative
% Memories (LAAM), otherwise they are Lattice HeteroAssociative Memories
% (LHAM).
%
% This implementation is optimized to run faster that EIA_LAM. It uses more memory.
%
% ------------------------------------------------------------------------------
% Input:   X         : input pattern matrix [nvariables x nsamples]
%          Y         : output pattern matrix [mvariables x nsamples]
%
% Output:  W         : dilative LAM [mvariables x nvariables]
%          M         : erosive LAM [mvariables x nvariables]
%
% Bibliographical references:
% [1] M. Gra??a, D. Chyzhyk, M. Garc??a-Sebasti??n, y C. Hern??ndez, ???Lattice independent component analysis for functional magnetic resonance imaging???, Information Sciences, vol. 181, n??. 10, p??gs. 1910-1928, May. 2011.
function [W,M] = EIA_LAM(X,Y)

%% Check parameters
if (nargin < 1)
    error('Insufficient parameters');
end
if (nargin < 2)
    Y = X;
end

% Checking data size
[nvariables kx] = size(X);
[mvariables ky] = size(Y);
if (kx ~= ky) || kx == 0
    error('Input and output dimensions missmatch or zero dimensionality');
end

%% Compute vector lattice external products
W = zeros(mvariables,nvariables);
M = zeros(mvariables,nvariables);
for b = 1:mvariables
    Yt = repmat(Y(b,:),nvariables,1);
    product = Yt - X;
    W(b,:) = min(product,[],2);
    M(b,:) = max(product,[],2);
end
W = W';
M = M';
